%
%  vel_pres_vtk2d(vtkfile, xy, t, vel, pressure)
%  vel_pres_vtk2d(vtkfile, xy, t, vel, pressure, vort, stream)
%
%
%  INPUT:   
%           vtkfile      base name of file
%                        vel_pres_vtk2d generates  vtkfile_vel.vtu
%                        and vel_pres_vtk2d_pres.vtu
%
%           xy           real nx x 2 
%                        x-y-coordinates of the vertices in the triangulation. 
%                        xy(i,1) x-coordinate of node i,
%                        xy(i,2) y-coordinate of node i.
% 
%           t            real nt x 6
%                        indices of the vertices in a triangle
%                        t(i,j) index of the j-th vertex in triangle i
%
%           vel          real nn x 2
%                        velocity at nodes
%
%           pressure     real nv x 1
%                        pressure at vertices
%
%           vort         real nn x 2
%                        vorticity at nodes [optional]
% 
%           stream       real nn x 2
%                        streamfunction at nodes [optional]
% 
%
%  AUTHOR:  Denis Ridzal and Matthias Heinkenschloss
%           Department of Computational and Applied Mathematics
%           Rice University
%           Novemer 15, 2005
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [] = vel_pres_vtk2d(vtkfile, xy, t, vel, pressure, varargin)


% Write pressures
% open file
filename = [vtkfile,'_pres.vtk'];
fid = fopen(filename,'w');

% write header
fprintf(fid, '%s\n', '# vtk DataFile Version 3.0');
fprintf(fid, '%s\n', 'created by vel_pres_vtk2d.m');
fprintf(fid, '%s\n', 'ASCII');  
fprintf(fid, '%s\n\n', 'DATASET UNSTRUCTURED_GRID');  

nv = max(max(t(:,1:3))); % number of vertices, assumes that vertices
                         % are first in xy, followed by mid points
                         % (true for meshes generated by triangle
fprintf(fid, '%s%d%s\n', 'POINTS ', nv, '  double');
for i = 1:nv
    fprintf(fid, '%13.6e  %13.6e   %13.6e \n', xy(i,:), 0);
end
fprintf(fid, '\n');
fprintf(fid, '%s %d %d\n', 'CELLS ', size(t,1), 4*size(t,1));
for i = 1:size(t,1)
    fprintf(fid, '%d  %d  %d  %d \n', 3, t(i,1:3)-1);
end
fprintf(fid, '\n');
fprintf(fid, '%s %d \n', 'CELL_TYPES ', size(t,1));
for i = 1:size(t,1)
    fprintf(fid, '%d \n', 5);
end
fprintf(fid, '\n');
fprintf(fid, '%s %d \n', 'POINT_DATA ', nv);
fprintf(fid, '%s \n', 'SCALARS pressure double 1');
fprintf(fid, '%s \n', 'LOOKUP_TABLE default');
for i = 1:nv
    fprintf(fid, '%13.6e \n', pressure(i));
end
fprintf(fid, '%s \n', 'VECTORS velocities double');
for i = 1:nv
    fprintf(fid, '%13.6e  %13.6e  %13.6e\n', vel(i,:), 0);
end

fclose(fid);
fprintf('... pressure data written to %s \n', filename)



% Write velocities
% open file
filename = [vtkfile,'_vel.vtk'];
fid = fopen(filename,'w');

% write header
fprintf(fid, '%s\n', '# vtk DataFile Version 3.0');
fprintf(fid, '%s\n', 'created by vel_pres_vtk2d.m');
fprintf(fid, '%s\n', 'ASCII');  
fprintf(fid, '%s\n\n', 'DATASET UNSTRUCTURED_GRID');  

fprintf(fid, '%s%d%s\n', 'POINTS ', size(xy,1), '  double');
for i = 1:size(xy,1)
    fprintf(fid, '%13.6e  %13.6e   %13.6e \n', xy(i,:), 0);
end
fprintf(fid, '\n');
fprintf(fid, '%s %d %d\n', 'CELLS ', 4*size(t,1), 16*size(t,1));
for i = 1:size(t,1)
    fprintf(fid, '%d  %d  %d  %d \n', 3, t(i,[1,6,5])-1);
    fprintf(fid, '%d  %d  %d  %d \n', 3, t(i,[4,5,6])-1);
    fprintf(fid, '%d  %d  %d  %d \n', 3, t(i,[2,4,6])-1);
    fprintf(fid, '%d  %d  %d  %d \n', 3, t(i,[5,4,3])-1);
end
fprintf(fid, '\n');
fprintf(fid, '%s %d \n', 'CELL_TYPES ', 4*size(t,1));
for i = 1:4*size(t,1)
    fprintf(fid, '%d \n', 5);
end
fprintf(fid, '\n');
fprintf(fid, '%s %d \n', 'POINT_DATA ', size(xy,1));
if( nargin > 5 )
    vort   = varargin{1};
    stream = varargin{2};
    fprintf(fid, '%s \n', 'SCALARS vorticity double 1');
    fprintf(fid, '%s \n', 'LOOKUP_TABLE default');
    for i = 1:size(xy,1)
        fprintf(fid, '%13.6e \n', vort(i));
    end
    fprintf(fid, '\n');
    fprintf(fid, '%s \n', 'SCALARS stream_fucntion double 1');
    fprintf(fid, '%s \n', 'LOOKUP_TABLE default');
    for i = 1:size(xy,1)
        fprintf(fid, '%13.6e \n', stream(i));
    end
end
fprintf(fid, '\n');
fprintf(fid, '%s \n', 'VECTORS velocities double');
for i = 1:size(xy,1)
    fprintf(fid, '%13.6e  %13.6e  %13.6e\n', vel(i,:), 0);
end

% close file
fclose(fid);

if( nargin > 5 )
    fprintf('... velocity, vorticity and streamfunction data written to %s \n', filename);
else
    fprintf('... velocity data written to %s \n', filename);
end


