/*
 * @(#)AnalysisModuleIOUTest.java
 *
 * Copyright (C) 2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.datastore;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import net.sourceforge.groboutils.autodoc.v1.AutoDoc;
import net.sourceforge.groboutils.codecoverage.v2.CCCreatorUtil;
import net.sourceforge.groboutils.codecoverage.v2.IAnalysisModule;


/**
 * Tests the AnalysisModuleIO class.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/04/15 05:48:28 $
 * @since     January 7, 2003
 */
public class AnalysisModuleIOUTest extends TestCase
{
    //-------------------------------------------------------------------------
    // Standard JUnit Class-specific declarations
    
    private static final Class THIS_CLASS = AnalysisModuleIOUTest.class;
    private static final AutoDoc DOC = new AutoDoc( THIS_CLASS );
    
    public AnalysisModuleIOUTest( String name )
    {
        super( name );
    }


    //-------------------------------------------------------------------------
    // Tests
    
    public void testWriteAnalysisModule1() throws Exception
    {
        AnalysisModuleIO amio = new AnalysisModuleIO();
        StringWriter sw = new StringWriter();
        IAnalysisModule am = createIAnalysisModule(
            "n", "u", "m" );
        
        amio.writeAnalysisModule( am, sw );
        
        String res = sw.toString();
        DOC.getLog().info( "Wrote module ["+res+"]" );
        assertEquals(
            "Incorrect result format.",
            "1:n,1:u,1:m",
            res );
    }
    
    public void testWriteAnalysisModule2() throws Exception
    {
        AnalysisModuleIO amio = new AnalysisModuleIO();
        StringWriter sw = new StringWriter();
        IAnalysisModule am = createIAnalysisModule(
            "", "", "" );
        
        amio.writeAnalysisModule( am, sw );
        
        String res = sw.toString();
        DOC.getLog().info( "Wrote module ["+res+"]" );
        assertEquals(
            "Incorrect result format.",
            "0:,0:,0:",
            res );
    }
    
    public void testWriteAnalysisModule3() throws Exception
    {
        AnalysisModuleIO amio = new AnalysisModuleIO();
        StringWriter sw = new StringWriter();
        IAnalysisModule am = createIAnalysisModule(
            "aaaa", "bbbb", "1234" );
        
        amio.writeAnalysisModule( am, sw );
        
        String res = sw.toString();
        DOC.getLog().info( "Wrote module ["+res+"]" );
        assertEquals(
            "Incorrect result format.",
            "4:aaaa,4:bbbb,4:1234",
            res );
    }
    
    
    public void testReadAnalysisModule1() throws Exception
    {
        AnalysisModuleIO amio = new AnalysisModuleIO();
        StringReader sr = new StringReader( "1:n,1:u,1:m" );
        IAnalysisModule am = amio.readAnalysisModule( sr );
        assertNotNull(
            "Returned null data.",
            am );
        assertEquals(
            "name incorrect.",
            "n",
            am.getMeasureName() );
        assertEquals(
            "unit incorrect.",
            "u",
            am.getMeasureUnit() );
        assertEquals(
            "instruction weight incorrect.",
            "m",
            am.getMimeEncoding() );
    }
    
    
    public void testReadAnalysisMetaData2() throws Exception
    {
        AnalysisModuleIO amio = new AnalysisModuleIO();
        StringReader sr = new StringReader( "0:   ,0:,0:asdf," );
        IAnalysisModule am = amio.readAnalysisModule( sr );
        assertNotNull(
            "Returned null data.",
            am );
        assertEquals(
            "name incorrect.",
            "",
            am.getMeasureName() );
        assertEquals(
            "unit incorrect.",
            "",
            am.getMeasureUnit() );
        assertEquals(
            "instruction weight incorrect.",
            "",
            am.getMimeEncoding() );
    }
    
    
    public void testReadAnalysisMetaData3() throws Exception
    {
        AnalysisModuleIO amio = new AnalysisModuleIO();
        StringReader sr = new StringReader( "" );
        try
        {
            amio.readAnalysisModule( sr );
            fail( "Did not throw IOException." );
        }
        catch (IOException ioe)
        {
            // verify ioe
        }
    }
    
    
    public void testReadAnalysisMetaData4() throws Exception
    {
        AnalysisModuleIO amio = new AnalysisModuleIO();
        StringReader sr = new StringReader( "1" );
        try
        {
            amio.readAnalysisModule( sr );
            fail( "Did not throw IOException." );
        }
        catch (IOException ioe)
        {
            // verify ioe
        }
    }
    
    
    public void testReadAnalysisMetaData5() throws Exception
    {
        AnalysisModuleIO amio = new AnalysisModuleIO();
        StringReader sr = new StringReader( "a:a,1:1,1:1" );
        try
        {
            amio.readAnalysisModule( sr );
            fail( "Did not throw IOException." );
        }
        catch (IOException ioe)
        {
            // verify ioe
        }
    }
    
    
    //-------------------------------------------------------------------------
    // Helpers
    
    protected IAnalysisModule createIAnalysisModule( String name, String unit,
            String mime )
    {
        return CCCreatorUtil.createIAnalysisModule( name, unit, mime );
    }
    
    
    //-------------------------------------------------------------------------
    // Standard JUnit declarations
    
    
    public static Test suite()
    {
        TestSuite suite = new TestSuite( THIS_CLASS );
        
        return suite;
    }
    
    public static void main( String[] args )
    {
        String[] name = { THIS_CLASS.getName() };
        
        // junit.textui.TestRunner.main( name );
        // junit.swingui.TestRunner.main( name );
        
        junit.textui.TestRunner.main( name );
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void setUp() throws Exception
    {
        super.setUp();
        
        // set ourself up
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void tearDown() throws Exception
    {
        // tear ourself down
        
        
        super.tearDown();
    }
}

