/***************************************************************************
 *   Copyright (C) 2005-2013 by Georg Hennig                               *
 *   Email: georg.hennig@web.de                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <QFileInfo>
#include <QImage>
#include <QLabel>
#include <QPushButton>
#include <QToolTip>
#include <QVBoxLayout>
#include <QScrollArea>
#include <QPixmap>

#include <KDebug>
#include <KIconLoader>
#include <kio/previewjob.h>
#include <KLocale>
#include <KProcess>
#include <KSeparator>
#include <KSqueezedTextLabel>

#include "kfileitemext.h"
#include "komparatorwidget.h"

#include "komparedialog.h"

#include <version.h>

KompareDialog::KompareDialog( QWidget *_parent, KFileItemExt *_item1, KFileItemExt *_item2,
	const QString &url1, const QString &url2, bool _md5 ) :
		KDialog( (QWidget *)_parent ),
		m_parent( _parent ), m_has_kdiff3( false ), m_md5( _md5 ), m_item1( _item1 ), m_item2( _item2 ), m_result( CANCEL )
{
	setCaption( i18n( "File compare dialog" ) );
	KDialog::setButtons( KDialog::Ok | KDialog::Cancel | KDialog::User1 );
	setModal( true );

	connect( this, SIGNAL( okClicked() ), this, SLOT( slotOk() ) );
	connect( this, SIGNAL( user1Clicked() ), this, SLOT( slotUser1() ) );

	m_kompare_proc = new KProcess( this );

	showButtonSeparator( true );
	showButton( KDialog::User1, false );
	setButtonText( KDialog::User1, i18n( "Compare with KDiff3" ) );
	setButtonToolTip( KDialog::User1, i18n( "Compare the two text files with KDiff3" ) );
	setButtonWhatsThis( KDialog::User1, i18n( "Compare the two text files with KDiff3" ) );

	if ( isTextFile( m_item1 ) && isTextFile( m_item2 ) )
	{
		*m_kompare_proc << "whereis" << "-b"/*binaries only*/ << "kdiff3";
		connect( m_kompare_proc, SIGNAL( readyReadStandardOutput() ), this, SLOT( slotReceivedStdout() ) );
		m_kompare_proc->setOutputChannelMode( KProcess::SeparateChannels );
		m_kompare_proc->start();
		connect( m_kompare_proc, SIGNAL( finished( int, QProcess::ExitStatus ) ), this, SLOT( slotProcessExited( int, QProcess::ExitStatus ) ) );
	}

	QWidget *main_h_box = new QWidget( this );
	QHBoxLayout *main_h_box_layout = new QHBoxLayout( main_h_box );
	main_h_box->setLayout( main_h_box_layout );

	QWidget *left_v_box = new QWidget( main_h_box );
	main_h_box_layout->addWidget( left_v_box );
	QVBoxLayout *left_v_box_layout = new QVBoxLayout( left_v_box );
	left_v_box->setLayout( left_v_box_layout );

	QWidget *center_v_box = new QWidget( main_h_box );
	main_h_box_layout->addWidget( center_v_box );
	QVBoxLayout *center_v_box_layout = new QVBoxLayout( center_v_box );
	center_v_box->setLayout( center_v_box_layout );

	QWidget *right_v_box = new QWidget( main_h_box );
	main_h_box_layout->addWidget( right_v_box );
	QVBoxLayout *right_v_box_layout = new QVBoxLayout( right_v_box );
	right_v_box->setLayout( right_v_box_layout );

	left_v_box_layout->addWidget( createTitle( left_v_box, m_item1 ) );

	left_v_box_layout->addWidget( new KSeparator( Qt::Horizontal, left_v_box ) );
	QWidget *label_column1 = createBasicInfos( left_v_box, m_item1, url1 );
	left_v_box_layout->addWidget( label_column1 );
	left_v_box_layout->addWidget( new KSeparator( Qt::Horizontal, left_v_box ) );
	QWidget *label_column2 = createExtendedInfos( left_v_box, m_item1 );
	left_v_box_layout->addWidget( label_column2 );

	left_v_box_layout->addWidget( new KSeparator( Qt::Horizontal, left_v_box ) );
	left_v_box_layout->addWidget( createMetaInfos( left_v_box, m_item1 ) );

	QSize small_size = 1.5 * SmallIcon( "edit-copy" ).size();

	center_v_box_layout->addWidget( new QLabel( "", center_v_box ) );
	m_copy_l_r = new QPushButton( center_v_box );
	center_v_box_layout->addWidget( m_copy_l_r );
	m_copy_l_r->setIconSize( QSize( small_size.width()*3, small_size.width() ) );
	m_copy_l_r->setIcon( ((KomparatorWidget*)m_parent)->broadIcon( SmallIcon( "edit-copy" ), SmallIcon( "arrow-right-double" ) ) );
	m_copy_l_r->setMaximumWidth( small_size.width()*3 );
	m_copy_l_r->setCheckable( true );
	connect( m_copy_l_r, SIGNAL( toggled( bool ) ), this, SLOT( slotCopyLeftRight( bool ) ) );
	m_copy_l_r->setToolTip( i18n( "Copy the file on the left to the right side" ) );
	m_copy_l_r->setWhatsThis( i18n( "Copy the file on the left to the right side" ) );

	m_move_l_r = new QPushButton( center_v_box );
	center_v_box_layout->addWidget( m_move_l_r );
	m_move_l_r->setIconSize( QSize( small_size.width()*3, small_size.width() ) );
	m_move_l_r->setIcon( ((KomparatorWidget*)m_parent)->broadIcon( SmallIcon( "go-jump" ), SmallIcon( "arrow-right-double" ) ) );
	m_move_l_r->setMaximumWidth( small_size.width()*3 );
	m_move_l_r->setCheckable( true );
	connect( m_move_l_r, SIGNAL( toggled( bool ) ), this, SLOT( slotMoveLeftRight( bool ) ) );
	m_move_l_r->setToolTip( i18n( "Move the file on the left to the right side" ) );
	m_move_l_r->setWhatsThis( i18n( "Move the file on the left to the right side" ) );

	m_trash_l = new QPushButton( center_v_box );
	center_v_box_layout->addWidget( m_trash_l );
	m_trash_l->setIconSize( QSize( small_size.width()*3, small_size.width() ) );
	m_trash_l->setIcon( ((KomparatorWidget*)m_parent)->broadIcon( SmallIcon( "arrow-right" ), SmallIcon( "user-trash" ) ) );
	m_trash_l->setMaximumWidth( small_size.width()*3 );
	m_trash_l->setCheckable( true );
	connect( m_trash_l, SIGNAL( toggled( bool ) ), this, SLOT( slotTrashLeft( bool ) ) );
	m_trash_l->setToolTip( i18n( "Trash the file on the left" ) );
	m_trash_l->setWhatsThis( i18n( "Trash the file on the left" ) );

	m_delete_l = new QPushButton( center_v_box );
	center_v_box_layout->addWidget( m_delete_l );
	m_delete_l->setIconSize( QSize( small_size.width()*3, small_size.width() ) );
	m_delete_l->setIcon( ((KomparatorWidget*)m_parent)->broadIcon( SmallIcon( "arrow-right" ), SmallIcon( "edit-delete" ) ) );
	m_delete_l->setMaximumWidth( small_size.width()*3 );
	m_delete_l->setCheckable( true );
	connect( m_delete_l, SIGNAL( toggled( bool ) ), this, SLOT( slotDeleteLeft( bool ) ) );
	m_delete_l->setToolTip( i18n( "Delete the file on the left" ) );
	m_delete_l->setWhatsThis( i18n( "Delete the file on the left" ) );

	QLabel *line = new QLabel( "", center_v_box );
	center_v_box_layout->addWidget( line );

	m_copy_r_l = new QPushButton( center_v_box );
	center_v_box_layout->addWidget( m_copy_r_l );
	m_copy_r_l->setIconSize( QSize( small_size.width()*3, small_size.width() ) );
	m_copy_r_l->setIcon( ((KomparatorWidget*)m_parent)->broadIcon( SmallIcon( "arrow-left-double" ), SmallIcon( "edit-copy" ) ) );
	m_copy_r_l->setMaximumWidth( small_size.width()*3 );
	m_copy_r_l->setCheckable( true );
	connect( m_copy_r_l, SIGNAL( toggled( bool ) ), this, SLOT( slotCopyRightLeft( bool ) ) );
	m_copy_r_l->setToolTip( i18n( "Copy the file on the right to the left side" ) );
	m_copy_r_l->setWhatsThis( i18n( "Copy the file on the right to the left side" ) );

	m_move_r_l = new QPushButton( center_v_box );
	center_v_box_layout->addWidget( m_move_r_l );
	m_move_r_l->setIconSize( QSize( small_size.width()*3, small_size.width() ) );
	m_move_r_l->setIcon( ((KomparatorWidget*)m_parent)->broadIcon( SmallIcon( "arrow-left-double" ), SmallIcon( "go-jump" ) ) );
	m_move_r_l->setMaximumWidth( small_size.width()*3 );
	m_move_r_l->setCheckable( true );
	connect( m_move_r_l, SIGNAL( toggled( bool ) ), this, SLOT( slotMoveRightLeft( bool ) ) );
	m_move_r_l->setToolTip( i18n( "Move the file on the right to the left side" ) );
	m_move_r_l->setWhatsThis( i18n( "Move the file on the right to the left side" ) );

	m_trash_r = new QPushButton( center_v_box );
	center_v_box_layout->addWidget( m_trash_r );
	m_trash_r->setIconSize( QSize( small_size.width()*3, small_size.width() ) );
	m_trash_r->setIcon( ((KomparatorWidget*)m_parent)->broadIcon( SmallIcon( "user-trash" ), SmallIcon( "arrow-left" ) ) );
	m_trash_r->setMaximumWidth( small_size.width()*3 );
	m_trash_r->setCheckable( true );
	connect( m_trash_r, SIGNAL( toggled( bool ) ), this, SLOT( slotTrashRight( bool ) ) );
	m_trash_r->setToolTip( i18n( "Trash the file on the right" ) );
	m_trash_r->setWhatsThis( i18n( "Trash the file on the right" ) );

	m_delete_r = new QPushButton( center_v_box );
	center_v_box_layout->addWidget( m_delete_r );
	m_delete_r->setIconSize( QSize( small_size.width()*3, small_size.width() ) );
	m_delete_r->setIcon( ((KomparatorWidget*)m_parent)->broadIcon( SmallIcon( "edit-delete" ), SmallIcon( "arrow-left" ) ) );
	m_delete_r->setMaximumWidth( small_size.width()*3 );
	m_delete_r->setCheckable( true );
	connect( m_delete_r, SIGNAL( toggled( bool ) ), this, SLOT( slotDeleteRight( bool ) ) );
	m_delete_r->setToolTip( i18n( "Delete the file on the right" ) );
	m_delete_r->setWhatsThis( i18n( "Delete the file on the right" ) );

	center_v_box_layout->addWidget( new QLabel( "", center_v_box ) );

	right_v_box_layout->addWidget( createTitle( right_v_box, m_item2 ) );
	right_v_box_layout->addWidget( new KSeparator( Qt::Horizontal, right_v_box ) );
	label_column1 = createBasicInfos( right_v_box, m_item2, url2 );
	right_v_box_layout->addWidget( label_column1 );
	right_v_box_layout->addWidget( new KSeparator( Qt::Horizontal, right_v_box ) );
	label_column2 = createExtendedInfos( right_v_box, m_item2 );
	right_v_box_layout->addWidget( label_column2 );

	right_v_box_layout->addWidget( new KSeparator( Qt::Horizontal, right_v_box ) );
	right_v_box_layout->addWidget( createMetaInfos( right_v_box, m_item2 ) );

	main_h_box->adjustSize();

	setMainWidget( main_h_box );

	KFileItemList item_list;
	item_list.append( *m_item1 );
	item_list.append( *m_item2 );
	KIO::PreviewJob *preview_job = new KIO::PreviewJob( item_list, 150/*width*/, 150/*height*/,
		48/*iconSize*/, 0/*iconAlpha*/, false/*scale*/, true/*save*/, NULL/*enabledPlugins*/ );
	connect( preview_job, SIGNAL( gotPreview( const KFileItem &, const QPixmap & ) ), this, SLOT( slotPreview( const KFileItem &, const QPixmap & ) ) );

	if ( width() > m_parent->width() ) resize( m_parent->width(), height() );
	if ( height() > m_parent->height() ) resize( width(), m_parent->height() );
}

KompareDialog::~KompareDialog()
{
}

int KompareDialog::getResult()
{
	return m_result;
}

QWidget *KompareDialog::createTitle( QWidget *local_parent, KFileItemExt *item )
{
	QWidget *title_box = new QWidget( local_parent );
	QHBoxLayout *title_box_layout = new QHBoxLayout( title_box );
	title_box->setLayout( title_box_layout );

	QLabel *title_pix = new QLabel( title_box );
	title_box_layout->addWidget( title_pix );
	title_pix->setPixmap( item->pixmap( 0 ) );
	KSqueezedTextLabel *title = new KSqueezedTextLabel( KIO::decodeFileName( item->url().fileName() ), title_box );
	title_box_layout->addWidget( title );
	title->setAlignment( Qt::AlignHCenter );

	return title_box;
}

QWidget *KompareDialog::createBasicInfos( QWidget *local_parent, KFileItemExt *item, const QString &url )
{
	QWidget *info_box = new QWidget( local_parent );
	QHBoxLayout *info_box_layout = new QHBoxLayout( info_box );
	info_box->setLayout( info_box_layout );

	QWidget *names_box = new QWidget( info_box );
	info_box_layout->addWidget( names_box );
	QVBoxLayout *names_box_layout = new QVBoxLayout( names_box );
	names_box->setLayout( names_box_layout );

	names_box_layout->addWidget( new QLabel( i18n( "Path:" ), names_box ) );
	names_box_layout->addWidget( new QLabel( i18n( "Size:" ), names_box ) );
	names_box_layout->addWidget( new QLabel( i18n( "Modified:" ), names_box ) );
	names_box_layout->addWidget( new QLabel( i18n( "Accessed:" ), names_box ) );
	names_box_layout->addWidget( new QLabel( i18n( "Owner:" ), names_box ) );
	names_box_layout->addWidget( new QLabel( i18n( "Group:" ), names_box ) );
	names_box_layout->addWidget( new QLabel( i18n( "Permissions:" ), names_box ) );

	QWidget *entries_box = new QWidget( info_box );
	info_box_layout->addWidget( entries_box );
	QVBoxLayout *entries_box_layout = new QVBoxLayout( entries_box );
	entries_box->setLayout( entries_box_layout );

	entries_box_layout->addWidget( new KSqueezedTextLabel( url, entries_box ) );
	QString size1_str = KIO::convertSize( item->size() );
	if ( item->size() >= 1024 ) size1_str += " (" + QString( "%1" ).arg( item->size() ) + " B)";
	entries_box_layout->addWidget( new QLabel( size1_str, entries_box ) );
	entries_box_layout->addWidget( new QLabel( item->getKFileItemExtModificationTimeString(), entries_box ) );
	entries_box_layout->addWidget( new QLabel( item->getKFileItemExtAccessTimeString(), entries_box ) );
	entries_box_layout->addWidget( new QLabel( item->user(), entries_box ) );
	entries_box_layout->addWidget( new QLabel( item->group(), entries_box ) );
	entries_box_layout->addWidget( new QLabel( item->permissionsString(), entries_box ) );

	return info_box;
}

QWidget *KompareDialog::createExtendedInfos( QWidget *local_parent, KFileItemExt *item )
{
	QString magicMimeComment  = QString::null;
	if ( item->isLocalFile() )
	{
		KMimeType::Ptr magicMimeType = KMimeType::findByFileContent( item->url().path() );
		if ( magicMimeType->name() != KMimeType::defaultMimeType() ) magicMimeComment = magicMimeType->comment();
	}

	QWidget *info_box = new QWidget( local_parent );
	QHBoxLayout *info_box_layout = new QHBoxLayout( info_box );
	info_box->setLayout( info_box_layout );

	QWidget *names_box = new QWidget( info_box );
	info_box_layout->addWidget( names_box );
	QVBoxLayout *names_box_layout = new QVBoxLayout( names_box );
	names_box->setLayout( names_box_layout );

	if ( item->isLink() ) names_box_layout->addWidget( new QLabel( i18n( "Points to:" ), names_box ) );
	names_box_layout->addWidget( new QLabel( i18n( "Type:" ), names_box ) );
	if ( !magicMimeComment.isEmpty() && magicMimeComment != item->mimeComment() ) names_box_layout->addWidget( new QLabel( i18n( "Contents:" ), names_box ) );
	names_box_layout->addWidget( new QLabel( i18n( "MD5 checksum:" ), names_box ) );

	QWidget *entries_box = new QWidget( info_box );
	info_box_layout->addWidget( entries_box );
	QVBoxLayout *entries_box_layout = new QVBoxLayout( entries_box );
	entries_box->setLayout( entries_box_layout );

	if ( item->isLink() ) entries_box_layout->addWidget( new KSqueezedTextLabel( item->linkDest(), entries_box ) );
	entries_box_layout->addWidget( new QLabel( item->mimeComment(), entries_box ) );
	if ( !magicMimeComment.isEmpty() && magicMimeComment != item->mimeComment() ) entries_box_layout->addWidget( new QLabel( magicMimeComment, entries_box ) );
	if ( m_md5 )
	{
		m_md5_label1 = new KSqueezedTextLabel( item->MD5(), entries_box );
		entries_box_layout->addWidget( m_md5_label1 );
	}
	else
	{
		QWidget *md5_box = new QWidget( entries_box );
		entries_box_layout->addWidget( md5_box );
		QVBoxLayout *md5_box_layout = new QVBoxLayout( md5_box );
		md5_box->setLayout( md5_box_layout );

		if ( item == m_item1 )
		{
			m_md5_button1 = new QPushButton( i18n( "Calculate" ), md5_box );
			md5_box_layout->addWidget( m_md5_button1 );
			connect( m_md5_button1, SIGNAL( clicked() ), SLOT( slotMD5_1() ) );
			m_md5_label1 = new KSqueezedTextLabel( md5_box );
			md5_box_layout->addWidget( m_md5_label1 );
			m_md5_label1->setMaximumWidth( 0 );
		}
		else
		{
			m_md5_button2 = new QPushButton( i18n( "Calculate" ), md5_box );
			md5_box_layout->addWidget( m_md5_button2 );
			connect( m_md5_button2, SIGNAL( clicked() ), SLOT( slotMD5_2() ) );
			m_md5_label2 = new KSqueezedTextLabel( md5_box );
			md5_box_layout->addWidget( m_md5_label2 );
			m_md5_label2->setMaximumWidth( 0 );
		}
	}

	return info_box;
}

void KompareDialog::slotMD5_1()
{
	m_md5_label1->setText( m_item1->MD5() );
	m_md5_label1->setMaximumWidth( 32767 );
	delete m_md5_button1;
	m_md5_button1 = NULL;
}

void KompareDialog::slotMD5_2()
{
	m_md5_label2->setText( m_item2->MD5() );
	m_md5_label2->setMaximumWidth( 32767 );
	delete m_md5_button2;
	m_md5_button2 = NULL;
}

QWidget *KompareDialog::createMetaInfos( QWidget *local_parent, KFileItemExt *item )
{
// 	QScrollArea *view = new QScrollArea( local_parent );
// 	view->setMinimumHeight( 120 );
// 	view->setSizePolicy( QSizePolicy::Expanding, QSizePolicy::Ignored );
// 	view->setFrameShape( Q3Frame::NoFrame );
// 	view->setResizePolicy( Q3ScrollView::AutoOneFit );

	QWidget *view_box = new QWidget( local_parent/*view->widget()*/ );
	QVBoxLayout *view_box_layout = new QVBoxLayout( view_box );
	view_box->setLayout( view_box_layout );

// 	view_box->setSpacing( 5 );
// 	view_box->setMargin( 5 );
// 	view->addChild( view_box );
// 	view->setWidget( view_box );

	if ( item == m_item1 )
	{
		m_preview_label1 = new QLabel( view_box );
		view_box_layout->addWidget( m_preview_label1 );
// 		m_preview_label1->setMaximumWidth( 0 );
// 		m_preview_label1->setMaximumHeight( 0 );
		m_preview_label1->setAlignment( Qt::AlignCenter );
	}
	else
	{
		m_preview_label2 = new QLabel( view_box );
		view_box_layout->addWidget( m_preview_label2 );
// 		m_preview_label2->setMaximumWidth( 0 );
// 		m_preview_label2->setMaximumHeight( 0 );
		m_preview_label2->setAlignment( Qt::AlignCenter );
	}

// 	QFileInfo file_info( item->url().path() );
// 	KFileMetaInfo m_info = item->metaInfo();
// 	if ( !m_info.isValid() )
// 	{
// 		m_info = KFileMetaInfo( item->url().path( KUrl::RemoveTrailingSlash ) );
// 		item->setMetaInfo( m_info );
// 	}
//
// 	if ( !m_info.isValid() || ( m_info.preferredKeys() ).isEmpty() ) return;

// FIXME: does that exist in KDE4?
// 	KFileMetaInfoProvider* prov = KFileMetaInfoProvider::self();
// 	KFileMetaInfoGroupList groupList = m_info.preferredGroups();
//
// 	KFileMimeTypeInfo* mtinfo = prov->mimeTypeInfo( m_info.mimeType() );
// 	if ( !mtinfo ) return;
//
// 	for ( KFileMetaInfoGroupList::Iterator git=groupList.begin(); git!=groupList.end(); ++git )
// 	{
// 		QStringList itemList = m_info.group(*git).preferredKeys();
// 		if (itemList.isEmpty()) continue;
//
// 		QGroupBox *group_box = new QGroupBox( 2, Qt::Horizontal, Q3StyleSheet::escape( mtinfo->groupInfo(*git)->translatedName() ), view_box );
//
// 		for ( QStringList::Iterator iit = itemList.begin(); iit!=itemList.end(); ++iit )
// 		{
// 			KFileMetaInfoItem item = m_info[*git][*iit];
// 			if ( !item.isValid() ) continue;
//
// 			QLabel *l = new QLabel( item.translatedKey() + ":", group_box );
// 			l->setAlignment( Qt::AlignLeft | Qt::AlignTop );
// 			new QLabel( item.string(), group_box );
// 		}
// 	}

	return view_box/*view*/;
}

void KompareDialog::slotPreview( const KFileItem &item, const QPixmap &pixmap )
{
	if ( item == *m_item1 )
	{
		m_preview_label1->setMaximumWidth( 32767 );
		m_preview_label1->setMaximumHeight( 32767 );
		m_preview_label1->setPixmap( pixmap );
	}
	else
	{
		m_preview_label2->setMaximumWidth( 32767 );
		m_preview_label2->setMaximumHeight( 32767 );
		m_preview_label2->setPixmap( pixmap );
	}
}

void KompareDialog::slotOk()
{
	m_result = 0;
	if ( m_copy_l_r->isChecked() ) m_result |= COPY_LEFT_TO_RIGHT;
	if ( m_move_l_r->isChecked() ) m_result |= MOVE_LEFT_TO_RIGHT;
	if ( m_trash_l->isChecked() )  m_result |= TRASH_LEFT;
	if ( m_delete_l->isChecked() ) m_result |= DELETE_LEFT;
	if ( m_copy_r_l->isChecked() ) m_result |= COPY_RIGHT_TO_LEFT;
	if ( m_move_r_l->isChecked() ) m_result |= MOVE_RIGHT_TO_LEFT;
	if ( m_trash_r->isChecked() )  m_result |= TRASH_RIGHT;
	if ( m_delete_r->isChecked() ) m_result |= DELETE_RIGHT;

	disconnect( this, SIGNAL( okClicked() ), this, SLOT( slotOk() ) );

	KDialog::slotButtonClicked( KDialog::Ok );
}

void KompareDialog::slotUser1()
{
	m_kompare_proc->kill();
	m_kompare_proc->setProgram( "kdiff3", QStringList() << m_item1->url().url() << m_item2->url().url() );

	if ( !( m_result & DIRTY ) ) // Only calculate md5 sum if files aren't dirty.
	{
		m_item1->MD5(); // calculate md5 sum now, so we can check, whether user modified something with kdiff3.
		m_item2->MD5(); // This should be rather fast, because text files aren't usually that large.
	}
	enableButton( KDialog::Ok, false );
	enableButton( KDialog::Cancel, false );
	enableButton( KDialog::User1, false );
	m_kompare_proc->start();
}

void KompareDialog::slotReceivedStdout()
{
	if ( !m_has_kdiff3 )
	{
		QString buffer_str = m_kompare_proc->readAllStandardOutput();
		buffer_str.remove( "kdiff3:" );
		if ( buffer_str.indexOf( "/kdiff3" ) != -1 )
		{
			m_has_kdiff3 = true;
			showButton( KDialog::User1, true );
		}
	}
}

void KompareDialog::slotProcessExited( int exitCode, QProcess::ExitStatus exitStatus )
{
	Q_UNUSED( exitCode );
	Q_UNUSED( exitStatus );

	if ( !( m_result & DIRTY ) ) // Only calculate md5 sum if files aren't dirty.
	{
		QByteArray md5_old1 = m_item1->MD5();
		QByteArray md5_old2 = m_item2->MD5();
		if ( md5_old1 != m_item1->MD5( true ) ||
				md5_old2 != m_item2->MD5( true ) )
		{
			m_result |= DIRTY;
		}
	}

	enableButton( KDialog::Ok, true );
	enableButton( KDialog::Cancel, true );
	enableButton( KDialog::User1, true );
}

void KompareDialog::slotCopyLeftRight( bool toggled )
{
	if ( toggled )
	{
		m_move_l_r->setChecked( false );
		m_trash_l->setChecked( false );
		m_delete_l->setChecked( false );

		m_copy_r_l->setChecked( false );
		m_move_r_l->setChecked( false );
		m_trash_r->setChecked( false );
		m_delete_r->setChecked( false );
	}
}

void KompareDialog::slotCopyRightLeft( bool toggled )
{
	if ( toggled )
	{
		m_copy_l_r->setChecked( false );
		m_move_l_r->setChecked( false );
		m_trash_l->setChecked( false );
		m_delete_l->setChecked( false );

		m_move_r_l->setChecked( false );
		m_trash_r->setChecked( false );
		m_delete_r->setChecked( false );
	}
}

void KompareDialog::slotMoveLeftRight( bool toggled )
{
	if ( toggled )
	{
		m_copy_l_r->setChecked( false );
		m_trash_l->setChecked( false );
		m_delete_l->setChecked( false );

		m_copy_r_l->setChecked( false );
		m_move_r_l->setChecked( false );
		m_trash_r->setChecked( false );
		m_delete_r->setChecked( false );
	}
}

void KompareDialog::slotMoveRightLeft( bool toggled )
{
	if ( toggled )
	{
		m_copy_l_r->setChecked( false );
		m_move_l_r->setChecked( false );
		m_trash_l->setChecked( false );
		m_delete_l->setChecked( false );

		m_copy_r_l->setChecked( false );
		m_trash_r->setChecked( false );
		m_delete_r->setChecked( false );
	}
}

void KompareDialog::slotTrashRight( bool toggled )
{
	if ( toggled )
	{
		m_copy_l_r->setChecked( false );
		m_move_l_r->setChecked( false );
		m_trash_l->setChecked( false );
		m_delete_l->setChecked( false );

		m_copy_r_l->setChecked( false );
		m_move_r_l->setChecked( false );
		m_delete_r->setChecked( false );
	}
}

void KompareDialog::slotTrashLeft( bool toggled )
{
	if ( toggled )
	{
		m_copy_l_r->setChecked( false );
		m_move_l_r->setChecked( false );
		m_delete_l->setChecked( false );

		m_copy_r_l->setChecked( false );
		m_move_r_l->setChecked( false );
		m_trash_r->setChecked( false );
		m_delete_r->setChecked( false );
	}
}

void KompareDialog::slotDeleteRight( bool toggled )
{
	if ( toggled )
	{
		m_copy_l_r->setChecked( false );
		m_move_l_r->setChecked( false );
		m_trash_l->setChecked( false );
		m_delete_l->setChecked( false );

		m_copy_r_l->setChecked( false );
		m_move_r_l->setChecked( false );
		m_trash_r->setChecked( false );
	}
}

void KompareDialog::slotDeleteLeft( bool toggled )
{
	if ( toggled )
	{
		m_copy_l_r->setChecked( false );
		m_move_l_r->setChecked( false );
		m_trash_l->setChecked( false );

		m_copy_r_l->setChecked( false );
		m_move_r_l->setChecked( false );
		m_trash_r->setChecked( false );
		m_delete_r->setChecked( false );
	}
}

bool KompareDialog::isTextFile( KFileItemExt *item )
{
	if ( item->mimetype().indexOf( "text" ) == 0 || item->mimetype() == "application/x-zerosize" )
	{
		return true;
	}

	KMimeType::Ptr magicMimeType = KMimeType::findByFileContent( item->url().path() );
	if ( magicMimeType->name().indexOf( "text" ) == 0 )
	{
		return true;
	}

	QStringList current_parent_mime_types = item->mimeTypePtr()->allParentMimeTypes();
	int i;
	for ( i=0; i<current_parent_mime_types.size(); i++ )
	{
		if ( current_parent_mime_types.at( i ).indexOf( "text" ) == 0 )
		{
			return true;
		}
	}

	QStringList magic_parent_mime_types = magicMimeType->allParentMimeTypes();
	for ( i=0; i<magic_parent_mime_types.size(); i++ )
	{
		if ( magic_parent_mime_types.at( i ).indexOf( "text" ) == 0 )
		{
			return true;
		}
	}

	return false;
}
